# pg_isok  A query centered monitoring tool for PostgreSQL
# Copyright (C) 2025 The Meme Factory, Inc., http://www.karlpinc.com/
#
#  This program is free software: you can redistribute it and/or modify
#  it under the terms of the GNU Affero General Public License as published
#  by the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU Affero General Public License for more details.
#
#  You should have received a copy of the GNU Affero General Public License
#  along with this program.  If not, see <https://www.gnu.org/licenses/>.
#
# Karl O. Pinc <kop@karlpinc.com>
#

# The latest supported version
LATEST_PG_VERSION := 18

#
# Use PGXS to build the extension
#
MODULES := 
DATA := sql/pg_isok--$(EXTVERSION).sql
# Sadly, PGXS seems to have no provision for installing html documentation
# that has a directory structure.  So, we (un)install that ourselves.
DOCS := doc/pg_isok.txt doc/pg_isok_a4.pdf doc/pg_isok_usletter.pdf \
        doc/pg_isok--$(EXTVERSION).config

#
# Select and patch tests for disabled features
#

ifdef DISABLE_ROLE
OMITTED_TESTS := 02057_run_resolved_updated_role \
                 02100_n_run_isok_queries_role \
                 02120_n_run_isok_queries_last_role
endif

ifdef DISABLE_SEARCH_PATH
OMITTED_TESTS += 02055_n_run_resolved_search_path \
                 02110_n_run_isok_queries_search_path \
                 02130_n_run_isok_queries_last_schemas
endif

# Determine the directory used for tests (and, hence, the test patches)
ifeq ($(origin DISABLE_ROLE) $(origin DISABLE_SEARCH_PATH), \
      undefined undefined)
# Both are enabled
PGXS_TEST_DIR := test/disable_none
else
ifneq ($(origin DISABLE_ROLE), undefined)
# Role is disabled
ifneq ($(origin DISABLE_SEARCH_PATH), undefined)
# Search path is also disabled
PGXS_TEST_DIR := test/disable_both
else
# Only role is disabled
PGXS_TEST_DIR := test/disable_role
endif
else
# Role is enabled
ifneq ($(origin DISABLE_SEARCH_PATH), undefined)
# Search path is disabled
PGXS_TEST_DIR := test/disable_search_path
endif
endif
endif

# Construct the tests to use, depending on what's disabled
ISOK_TEST_SRC := $(filter-out $(patsubst %,test/sql/%.sql,$(OMITTED_TESTS)), \
                              $(wildcard test/sql/*.sql))
ISOK_EXPECTED_SRC := $(patsubst %.sql,%.out, \
                       $(patsubst test/sql/%,test/expected/%,$(ISOK_TEST_SRC)))
.PHONY: installcheck_disabled
installcheck_disabled:
	rm -rf $(PGXS_TEST_DIR)/sql $(PGXS_TEST_DIR)/expected
	mkdir -p $(PGXS_TEST_DIR)/sql $(PGXS_TEST_DIR)/expected
	cp $(ISOK_TEST_SRC) $(PGXS_TEST_DIR)/sql
	cp $(ISOK_EXPECTED_SRC) $(PGXS_TEST_DIR)/expected
	[ ! -e $(PGXS_TEST_DIR)/adjust_linenos.patch ] \
	  || patch -p0 < $(PGXS_TEST_DIR)/adjust_linenos.patch

installcheck: installcheck_disabled

#
# Patch tests for PG version
#

PG_CONFIG := pg_config

THIS_PG_VERSION := $(shell $(PG_CONFIG) --version \
                           | cut -d ' ' -f 2 \
                           | cut -d . -f 1)

.PHONY: installcheck_patched
installcheck_patched:
	pgv=$$(($(LATEST_PG_VERSION) - 1)) ; \
	while [ $$pgv -ge $(THIS_PG_VERSION) ] ; do \
	  case $(PGXS_TEST_DIR) in \
	    test/disable_none) \
	      patch_file=test/version/$$pgv/disable_none.patch ; \
	      ;; \
	    test/disable_role) \
	      patch_file=test/version/$$pgv/disable_role.patch ; \
	      ;; \
	    test/disable_search_path) \
	      patch_file=test/version/$$pgv/disable_search_path.patch ; \
	      ;; \
	    test/disable_both) \
	      patch_file=test/version/$$pgv/disable_both.patch ; \
	      ;; \
	  esac ; \
	  if [ -e $$patch_file ] ; then \
	    patch -p0 < $$patch_file ; \
	  fi ; \
	  pgv=$$(($$pgv - 1)) ; \
	done

installcheck_patched: installcheck_disabled

installcheck: installcheck_patched

REGRESS = $(patsubst $(PGXS_TEST_DIR)/sql/%.sql,%, \
                     $(sort $(wildcard $(PGXS_TEST_DIR)/sql/*.sql)))
REGRESS_OPTS := --inputdir=$(PGXS_TEST_DIR)

PGXS := $(shell $(PG_CONFIG) --pgxs)
include $(PGXS)

#
# Do the install/uninstall of the html docs
#
PG_ISOK_HTML_DOC_DIR := '$(DESTDIR)$(docdir)/$(docmoduledir)/pg_isok_html/'

install_pg_isok_html:
	$(MKDIR_P) $(PG_ISOK_HTML_DOC_DIR)
	cd doc/pg_isok_html \
	; find . -type f -exec \
	        $(INSTALL_DATA) -D $(addprefix $(srcdir)/, \{\}) \
	                        $(PG_ISOK_HTML_DOC_DIR)/\{\} \;

install: install_pg_isok_html

uninstall_pg_isok_html:
	rm -rf $(PG_ISOK_HTML_DOC_DIR)

uninstall: uninstall_pg_isok_html

#
# Maintenance targets
#

# Clean up the generated regression tests
.PHONY: clean_pgxs
clean_pgxs:
	rm -rf test/disable_none \
	       test/disable_role/sql test/disable_role/expected \
	       test/disable_search_path/sql test/disable_search_path/expected \
	       test/disable_both/sql test/disable_both/expected

clean: clean_pgxs
