# Copyright (C) 2012, 2015, 2024, 2025 The Meme Factory, Inc.
#   http://www.karlpinc.com/
# Copyright (C) 2022, 2024 Jake Gordon <jacob.gordon@duke.edu>
# Copyright (C) 2004, 2005, 2008, 2011 Karl O. Pinc  <kop@karlpinc.com>
#
#  This program is free software: you can redistribute it and/or modify
#  it under the terms of the GNU Affero General Public License as published
#  by the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU Affero General Public License for more details.
#
#  You should have received a copy of the GNU Affero General Public License
#  along with this program.  If not, see <https://www.gnu.org/licenses/>.
#
# Makefile for database content
#
# Karl O. Pinc <kop@karlpinc.com>
#

##
## ##################################################################
##                                 SQL
##

## Variables:
##     TARGET_SCHEMA Used only when making a SQL file suitable for
##                   manual installation in the cloud.  The schema in
##                   which to install all database objects.
##     DISABLE_ROLE  See above.
##     DISABLE_SEARCH_PATH
##                   See above.
##
## Targets:
# It is tempting to set a default value in the Makefile, but explicit
# is better than implicit.  No surprises that way.

SQL_PATH := sql
# The SQL script files that are built for the extension
SQL_SCRIPTS := $(SQL_PATH)/pg_isok--$(EXTVERSION).sql

# Add the sql we always want to generate to the "all" target:
all: $(SQL_SCRIPTS)

DB_PATH := db
INCLUDE_PATH := $(DB_PATH)/include
FUNCTIONS_PATH := $(DB_PATH)/functions
TRIGGERS_PATH := $(DB_PATH)/triggers

# DISABLE_* state
DISABLE_ROLE_CACHE := $(DB_PATH)/disable_role_used
DISABLE_SEARCH_PATH_CACHE := $(DB_PATH)/disable_search_path_used
CONFIG_DOC := $(BUILT_PATH)/pg_isok--$(EXTVERSION).config

FUNCTION_SOURCES := $(wildcard $(FUNCTIONS_PATH)/*.m4)
TRIGGER_SOURCES := $(wildcard $(TRIGGERS_PATH)/*.m4)

DB_TARGETS := $(patsubst %,$(DB_PATH)/%, \
                         maketables \
                         makeindexes)
GENERATED_DB_TARGETS := $(DB_PATH)/maketables
FUNCTION_TARGETS := $(patsubst %.m4,%,$(FUNCTION_SOURCES))
TRIGGER_TARGETS := $(patsubst %.m4,%,$(TRIGGER_SOURCES))
SQL_TARGETS := $(DB_PATH)/pg_isok--$(EXTVERSION).sql \
               $(SQL_SCRIPTS)
CLOUD_TARGETS := $(SQL_PATH)/pg_isok_cloud--$(EXTVERSION).sql

MAKEDEPENDS_INCLUDES := $(wildcard $(INCLUDE_PATH)/*.m4)
MAKEDEPENDS_DB := $(MAKEDEPENDS) $(MAKEDEPENDS_INCLUDES)

#
# Generate table stuff
#
$(DB_PATH)/maketables: $(DB_PATH)/maketables.m4 \
                           $(INCLUDE_PATH)/copyright.m4 \
                           $(INCLUDE_PATH)/tablemacros.m4 \
                           $(MAKEDEPENDS_DB)
	m4 -I $(INCLUDE_PATH) $(DB_PATH)/maketables.m4 \
	  > $(DB_PATH)/maketables

#
# Generate function stuff
#

# Save the DISABLE_ROLE state, so we don't have to always rebuild the
# functions just in case the value changes.  This means that we can
# install without rebuilding the functions, so regular users who
# install the PGXN distribution won't need m4 installed.  Similarly,
# it means that invoking the install target uses the DISABLE_ROLE set
# when the sql was built.  Invocations of install do not need to
# re-specify DISABLE_ROLE.

ifdef DISABLE_ROLE
M4_DISABLE_ROLE := --define isok_disable_role
endif

ifdef DISABLE_SEARCH_PATH
M4_DISABLE_SEARCH_PATH := --define isok_disable_search_path
endif

ifneq ($(MAKECMDGOALS),install)
# Any target but "install", update the DISABLE_* caches

# Kluge that uses shell to always run some shell code when make is run
# This updates the DISABLE_ROLE_CACHE when DISABLE_ROLE changes
ignored_output := $(shell \
  { { [ ! -e $(DISABLE_ROLE_CACHE) ] \
      || [ "$$(cat $(DISABLE_ROLE_CACHE))" != "$(M4_DISABLE_ROLE)" ] ; } \
    && printf -- '$(M4_DISABLE_ROLE)' > $(DISABLE_ROLE_CACHE) ; } \
  || true \
  ; { { [ ! -e $(DISABLE_SEARCH_PATH_CACHE) ] \
        || [ "$$(cat $(DISABLE_SEARCH_PATH_CACHE))" \
             != "$(M4_DISABLE_SEARCH_PATH)" ] ; } \
      && printf -- '$(M4_DISABLE_SEARCH_PATH)' \
           > $(DISABLE_SEARCH_PATH_CACHE) ; } \
    || true \
  ; mkdir -p $(BUILT_PATH) \
  ; printf 'Built with:\n' > $(CONFIG_DOC) \
  ; printf '  DISABLE_ROLE=$(DISABLE_ROLE)\n' >> $(CONFIG_DOC) \
  ; printf '  DISABLE_SEARCH_PATH=$(DISABLE_SEARCH_PATH)\n' >> $(CONFIG_DOC) \
  ; printf '  TARGET_SCHEMA=$(TARGET_SCHEMA)\n' >> $(CONFIG_DOC) \
) $(info Updating $(DISABLE_ROLE_CACHE) to "$(M4_DISABLE_ROLE)") \
  $(info Updating $(DISABLE_SEARCH_PATH_CACHE) to "$(M4_DISABLE_SEARCH_PATH)"))
endif

$(FUNCTION_TARGETS): %: %.m4 $(DISABLE_ROLE_CACHE) $(MAKEDEPENDS_DB)
	m4 $(shell cat $(DISABLE_ROLE_CACHE)) \
	   $(shell cat $(DISABLE_SEARCH_PATH_CACHE)) \
	   -I $(INCLUDE_PATH) $< > $@

#
# Generate trigger stuff
#
$(TRIGGER_TARGETS): %: %.m4 $(MAKEDEPENDS_DB)
	m4 -I $(INCLUDE_PATH) $< > $@
#
# Generate in-database comments
#
include $(MAKE_FILES)/comments.mk

#
# Generate the script file
#
$(SQL_PATH): $(MAKEDEPENDS_DB)
	mkdir -p $(SQL_PATH)

# Generate an SQL file that will be used to generate the final SQL.
# This intermediate step allows us to generate both the SQL used
# by CREATE EXTENSION and SQL that can be manually installed.
$(DB_PATH)/pg_isok--$(EXTVERSION).sql: \
                            $(DB_TARGETS) $(TRIGGER_TARGETS) \
                            $(FUNCTION_TARGETS) $(COMMENTS_TARGETS) \
                            $(MAKEDEPENDS_DB)
	cat $(DB_TARGETS) $(TRIGGER_TARGETS) $(FUNCTION_TARGETS) \
            $(COMMENTS_TARGETS) > $@

##   sql/pg_isok--VERSION.sql
##       Generate the file of SQL used by CREATE EXTENSION
##       (where VERSION is the pg_isok version)
$(SQL_SCRIPTS): \
                            $(DB_PATH)/pg_isok--$(EXTVERSION).sql \
                            $(DB_PATH)/config_dump.sql \
                            $(MAKEDEPENDS_DB) \
                            | $(SQL_PATH)
	cat $< $(DB_PATH)/config_dump.sql > $@

##   sql/pg_isok_cloud--VERSION.sql
##       Generate a file of SQL that can be executed "by hand".
##       The TARGET_SCHEMA variable must be set and is where the
##       extension's objects are installed.  For example:
##         make TARGET_SCHEMA=myschema sql/pg_isok_cloud--VERSION.sql
##       See the installation documentation for further detail.
# Always re-generate the file, even if it exists, because the TARGET_SCHEMA
# may change.
.PHONY: $(SQL_PATH)/pg_isok_cloud--$(EXTVERSION).sql
$(SQL_PATH)/pg_isok_cloud--$(EXTVERSION).sql: \
                         $(DB_PATH)/pg_isok--$(EXTVERSION).sql \
                         $(MAKEDEPENDS_DB)
	[ -n "$(TARGET_SCHEMA)" ] \
	  || { printf 'The TARGET_SCHEMA variable must be set\n' >&2 ; \
	       exit 1 ; }
	echo 'BEGIN TRANSACTION;' > $@
	echo 'SET LOCAL search_path TO "$(TARGET_SCHEMA)", pg_temp;' >> $@
	sed -e 's/@extschema@/"$(TARGET_SCHEMA)"/g' $< >> $@
	echo 'COMMIT TRANSACTION;' >> $@

#
# Maintenance targets
#

.PHONY: maintainer-clean_db
maintainer-clean_db: clean_db
	rm -rf $(SQL_TARGETS) $(GENERATED_DB_TARGETS) \
	       $(FUNCTION_TARGETS) $(TRIGGER_TARGETS) $(SQL_PATH) \
	       $(DISABLE_ROLE_CACHE) $(DISABLE_SEARCH_PATH_CACHE)

.PHONY: clean_db
clean_db:
	rm -rf $(CLOUD_TARGETS)

.PHONY: clean
clean: clean_db

.PHONY: maintainer-clean
maintainer-clean: maintainer-clean_db
